# Library Interface Contract

This document describes the interface contract between DetectMateService and DetectMateLibrary. If you're implementing custom components in the library, your classes must adhere to these interfaces.

## CoreComponent Interface

All processing components (readers, parsers, detectors) must inherit from `CoreComponent`:

```python
from detectmatelibrary.common.core import CoreComponent

class MyComponent(CoreComponent):
    def __init__(self, config=None):
        """Initialize the component.

        Args:
            config: Optional configuration dictionary or CoreConfig instance.
                    May be None if no configuration is provided.
        """
        super().__init__(config)
        # Your initialization here

    def process(self, data: bytes) -> bytes | None:
        """Process incoming data.

        Args:
            data: Raw bytes received from the upstream component.

        Returns:
            bytes: Processed data to forward to downstream components.
            None: Skip forwarding (filter out this message).
        """
        # Your processing logic here
        return processed_data
```

### Key Requirements

- **Constructor**: Must accept an optional `config` parameter (can be `dict` or `CoreConfig` instance)
- **process() method**: Must accept `bytes` and return `bytes | None`
- **Return behavior**:
  - Return `bytes` to forward output to downstream components
  - Return `None` to skip/filter the message (no output sent)

## CoreConfig Interface

Configuration classes must inherit from `CoreConfig`, which extends Pydantic's `BaseModel`:

```python
from detectmatelibrary.common.core import CoreConfig

class MyComponentConfig(CoreConfig):
    """Configuration for MyComponent."""
    threshold: float = 0.5
    window_size: int = 10
    enabled: bool = True
```

### Key Requirements

- **Pydantic BaseModel**: Must support `model_validate()` and `model_dump()` methods
- **Type hints**: All fields should have type annotations
- **Defaults**: Provide sensible defaults where appropriate

### Configuration Flow

1. Service loads config from YAML file via `ConfigManager`
2. Config class validates the data using `model_validate()`
3. Validated config is passed to component constructor
4. At runtime, `reconfigure` command can update configs dynamically

## Component Loading

Components are loaded dynamically by `ComponentLoader`. Specify components using a dot-separated path:

### Path Format

```
module.ClassName
```

Examples:
- `detectors.RandomDetector`
- `parsers.JsonParser`
- `readers.FileReader`

### Resolution Order

1. **DetectMateLibrary-relative** (tried first): `detectmatelibrary.{path}`
   - `detectors.RandomDetector` → `detectmatelibrary.detectors.RandomDetector`
2. **Absolute import** (fallback): `{path}` as-is
   - `mypackage.detectors.CustomDetector` → `mypackage.detectors.CustomDetector`

This allows you to use library components with short paths while still supporting custom components from external packages.

### Service Settings

In your service settings YAML, specify:

```yaml
component_type: detectors.MyDetector          # Component class path
component_config_class: detectors.MyDetectorConfig  # Config class path
config_file: detector-config.yaml             # Path to component config
```

## Data Flow Schemas

Components in the processing pipeline use protobuf schemas for structured data exchange:

| Stage | Input | Output Schema |
|-------|-------|---------------|
| Reader | Raw source (file, network, etc.) | `LogSchema` |
| Parser | `LogSchema` bytes | `ParserSchema` |
| Detector | `ParserSchema` bytes | `DetectorSchema` |

Each component receives serialized protobuf bytes, deserializes them, processes the data, and serializes the output for the next stage.

## Complete Example

Here's a minimal detector component implementation:

### 1. Config Class (`detectors/threshold_detector.py`)

```python
from detectmatelibrary.common.core import CoreConfig, CoreComponent

class ThresholdDetectorConfig(CoreConfig):
    """Configuration for threshold-based anomaly detection."""
    threshold: float = 0.8
    alert_on_exceed: bool = True

class ThresholdDetector(CoreComponent):
    """Detects anomalies when values exceed a threshold."""

    def __init__(self, config=None):
        super().__init__(config)

        # Handle config as dict or CoreConfig
        if config is None:
            self.threshold = 0.8
            self.alert_on_exceed = True
        elif isinstance(config, dict):
            self.threshold = config.get('threshold', 0.8)
            self.alert_on_exceed = config.get('alert_on_exceed', True)
        else:
            self.threshold = config.threshold
            self.alert_on_exceed = config.alert_on_exceed

    def process(self, data: bytes) -> bytes | None:
        """Process incoming data and detect anomalies.

        Args:
            data: Serialized ParserSchema bytes

        Returns:
            Serialized DetectorSchema bytes if anomaly detected,
            None if no anomaly
        """
        # Deserialize input (example with protobuf)
        # parser_output = ParserSchema()
        # parser_output.ParseFromString(data)

        # Your detection logic here
        value = self._extract_value(data)

        if self._is_anomaly(value):
            # Create and return detector output
            return self._create_alert(data, value)

        return None  # No anomaly, skip forwarding

    def _extract_value(self, data: bytes) -> float:
        """Extract numeric value from data."""
        # Implementation depends on your schema
        return 0.0

    def _is_anomaly(self, value: float) -> bool:
        """Check if value exceeds threshold."""
        if self.alert_on_exceed:
            return value > self.threshold
        return value < self.threshold

    def _create_alert(self, data: bytes, value: float) -> bytes:
        """Create alert output."""
        # Serialize DetectorSchema output
        return data  # Simplified example
```

### 2. Service Settings (`settings.yaml`)

```yaml
component_name: threshold-detector
component_type: detectors.threshold_detector.ThresholdDetector
component_config_class: detectors.threshold_detector.ThresholdDetectorConfig
config_file: threshold-config.yaml
log_level: INFO
http_host: 127.0.0.1
http_port: 8000
engine_addr: ipc:///tmp/threshold.engine.ipc
```

### 3. Component Config (`threshold-config.yaml`)

Component configuration uses a nested, namespaced structure:

```yaml
detectors:                    # Category level
  ThresholdDetector:          # Class name level
    threshold: 0.9
    alert_on_exceed: true
```

This hierarchical format allows the library to correctly route parameters based on category and class name.

### 4. Run the Service

```bash
detectmate start --settings settings.yaml
```

## Validation

The service validates components at load time:

1. **Component class**: Must be an instance of `CoreComponent`
2. **Config class**: Must be a subclass of `CoreConfig`

If validation fails, the service raises an error with a descriptive message.
